pragma solidity ^0.8.24;

import "forge-std/Test.sol";
import {TicketRegistry} from "../src/TicketRegistry.sol";

contract TicketRegistryTest is Test {
    TicketRegistry reg;

    address owner = address(0xABCD);
    address alice = address(0xA11CE);
    address bob = address(0xB0B);

    function setUp() public {
        vm.prank(owner);
        reg = new TicketRegistry(owner);
    }

    function testCreateTicket_OnlyOwner() public {
        bytes32 h = keccak256(abi.encodePacked("canonical-json"));
        // non-owner should revert
        vm.expectRevert(); // "Ownable: caller is not the owner"
        reg.createTicket(h, 2, "Facilities");

        // owner can create
        vm.startPrank(owner);
        uint256 id = reg.createTicket(h, 2, "Facilities");
        vm.stopPrank();

        assertEq(id, 1);
        (
            bytes32 hash_,
            uint8 severity_,
            string memory dept_,
            address creator_,
            address assignee_,
            TicketRegistry.Status status_,
            uint64 createdAt_,
            uint64 updatedAt_
        ) = reg.getTicket(id);

        assertEq(hash_, h);
        assertEq(severity_, 2);
        assertEq(dept_, "Facilities");
        assertEq(creator_, owner);
        assertEq(assignee_, address(0));
        assertEq(uint(status_), uint(TicketRegistry.Status.Open));
        assertGt(createdAt_, 0);
        assertEq(createdAt_, updatedAt_);
        assertEq(reg.totalTickets(), 1);
        assertTrue(reg.exists(1));
    }

    function testAssign_OnlyOwner() public {
        bytes32 h = keccak256("a");
        vm.prank(owner);
        uint256 id = reg.createTicket(h, 1, "Electrical");

        // non-owner cannot assign
        vm.prank(alice);
        vm.expectRevert(); // Ownable revert
        reg.assignTicket(id, bob);

        // owner assigns
        vm.prank(owner);
        reg.assignTicket(id, bob);

        (, , , , address assignee, , , ) = reg.getTicket(id);
        assertEq(assignee, bob);
    }

    function testUpdateStatus_OwnerOrAssignee() public {
        bytes32 h = keccak256("x");
        vm.prank(owner);
        uint256 id = reg.createTicket(h, 3, "Plumbing");

        // not owner/assignee -> revert
        vm.prank(alice);
        vm.expectRevert(bytes("not authorized"));
        reg.updateStatus(id, TicketRegistry.Status.InProgress);

        // owner can update
        vm.prank(owner);
        reg.updateStatus(id, TicketRegistry.Status.InProgress);

        // assign to bob
        vm.prank(owner);
        reg.assignTicket(id, bob);

        // assignee can update
        vm.prank(bob);
        reg.updateStatus(id, TicketRegistry.Status.Resolved);

        (, , , , , TicketRegistry.Status status, , ) = reg.getTicket(id);
        assertEq(uint(status), uint(TicketRegistry.Status.Resolved));
    }

    function testCreateRequiresHash() public {
        vm.prank(owner);
        vm.expectRevert(bytes("hash required"));
        reg.createTicket(bytes32(0), 0, "Any");
    }

    function testNonexistentTicketReverts() public {
        vm.expectRevert(bytes("ticket not found"));
        reg.getTicket(999);
    }
}
